﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/application-signals/ApplicationSignalsRequest.h>
#include <aws/application-signals/ApplicationSignals_EXPORTS.h>
#include <aws/application-signals/model/AuditTarget.h>
#include <aws/application-signals/model/DetailLevel.h>
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace ApplicationSignals {
namespace Model {

/**
 */
class ListAuditFindingsRequest : public ApplicationSignalsRequest {
 public:
  AWS_APPLICATIONSIGNALS_API ListAuditFindingsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListAuditFindings"; }

  AWS_APPLICATIONSIGNALS_API Aws::String SerializePayload() const override;

  AWS_APPLICATIONSIGNALS_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The start of the time period to retrieve audit findings for. When used in a
   * raw HTTP Query API, it is formatted as epoch time in seconds. For example,
   * <code>1698778057</code> </p>
   */
  inline const Aws::Utils::DateTime& GetStartTime() const { return m_startTime; }
  inline bool StartTimeHasBeenSet() const { return m_startTimeHasBeenSet; }
  template <typename StartTimeT = Aws::Utils::DateTime>
  void SetStartTime(StartTimeT&& value) {
    m_startTimeHasBeenSet = true;
    m_startTime = std::forward<StartTimeT>(value);
  }
  template <typename StartTimeT = Aws::Utils::DateTime>
  ListAuditFindingsRequest& WithStartTime(StartTimeT&& value) {
    SetStartTime(std::forward<StartTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The end of the time period to retrieve audit findings for. When used in a raw
   * HTTP Query API, it is formatted as epoch time in seconds. For example,
   * <code>1698778057</code> </p>
   */
  inline const Aws::Utils::DateTime& GetEndTime() const { return m_endTime; }
  inline bool EndTimeHasBeenSet() const { return m_endTimeHasBeenSet; }
  template <typename EndTimeT = Aws::Utils::DateTime>
  void SetEndTime(EndTimeT&& value) {
    m_endTimeHasBeenSet = true;
    m_endTime = std::forward<EndTimeT>(value);
  }
  template <typename EndTimeT = Aws::Utils::DateTime>
  ListAuditFindingsRequest& WithEndTime(EndTimeT&& value) {
    SetEndTime(std::forward<EndTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of auditor names to filter the findings by. Only findings generated by
   * the specified auditors will be returned.</p> <p>The following auditors are
   * available for configuration:</p> <ul> <li> <p> <code>slo</code> - SloAuditor:
   * Identifies SLO violations and detects breached thresholds during the Assessment
   * phase.</p> </li> <li> <p> <code>operation_metric</code> -
   * OperationMetricAuditor: Detects anomalies in service operation metrics from
   * Application Signals RED metrics during the Assessment phase</p>
   * <p>Anomaly detection is not supported for sparse metrics (those missing more
   * than 80% of datapoints within the given time period).</p>  </li> <li> <p>
   * <code>service_quota</code> - ServiceQuotaAuditor: Monitors resource utilization
   * against service quotas during the Assessment phase</p> </li> <li> <p>
   * <code>trace</code> - TraceAuditor: Performs deep-dive analysis of distributed
   * traces, correlating traces with breached SLOs or abnormal RED metrics during the
   * Analysis phase</p> </li> <li> <p> <code>dependency_metric</code> -
   * CriticalPathAuditor: Analyzes service dependency impacts and maps dependency
   * relationships from Application Signals RED metrics during the Analysis phase</p>
   * </li> <li> <p> <code>top_contributor</code> - TopContributorAuditor: Identifies
   * infrastructure-level contributors to issues by analyzing EMF logs of Application
   * Signals RED metrics during the Analysis phase</p> </li> <li> <p>
   * <code>log</code> - LogAuditor: Extracts insights from application logs,
   * categorizing error types and ranking severity by frequency during the Analysis
   * phase</p> </li> </ul>  <p> <code>InitAuditor</code> and
   * <code>Summarizer</code> auditors are not configurable as they are automatically
   * triggered during the audit process.</p>
   */
  inline const Aws::Vector<Aws::String>& GetAuditors() const { return m_auditors; }
  inline bool AuditorsHasBeenSet() const { return m_auditorsHasBeenSet; }
  template <typename AuditorsT = Aws::Vector<Aws::String>>
  void SetAuditors(AuditorsT&& value) {
    m_auditorsHasBeenSet = true;
    m_auditors = std::forward<AuditorsT>(value);
  }
  template <typename AuditorsT = Aws::Vector<Aws::String>>
  ListAuditFindingsRequest& WithAuditors(AuditorsT&& value) {
    SetAuditors(std::forward<AuditorsT>(value));
    return *this;
  }
  template <typename AuditorsT = Aws::String>
  ListAuditFindingsRequest& AddAuditors(AuditorsT&& value) {
    m_auditorsHasBeenSet = true;
    m_auditors.emplace_back(std::forward<AuditorsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of audit targets to filter the findings by. You can specify services,
   * SLOs, or service operations to limit the audit findings to specific
   * entities.</p>
   */
  inline const Aws::Vector<AuditTarget>& GetAuditTargets() const { return m_auditTargets; }
  inline bool AuditTargetsHasBeenSet() const { return m_auditTargetsHasBeenSet; }
  template <typename AuditTargetsT = Aws::Vector<AuditTarget>>
  void SetAuditTargets(AuditTargetsT&& value) {
    m_auditTargetsHasBeenSet = true;
    m_auditTargets = std::forward<AuditTargetsT>(value);
  }
  template <typename AuditTargetsT = Aws::Vector<AuditTarget>>
  ListAuditFindingsRequest& WithAuditTargets(AuditTargetsT&& value) {
    SetAuditTargets(std::forward<AuditTargetsT>(value));
    return *this;
  }
  template <typename AuditTargetsT = AuditTarget>
  ListAuditFindingsRequest& AddAuditTargets(AuditTargetsT&& value) {
    m_auditTargetsHasBeenSet = true;
    m_auditTargets.emplace_back(std::forward<AuditTargetsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The level of details of the audit findings. Supported values:
   * <code>BRIEF</code>, <code>DETAILED</code>.</p>
   */
  inline DetailLevel GetDetailLevel() const { return m_detailLevel; }
  inline bool DetailLevelHasBeenSet() const { return m_detailLevelHasBeenSet; }
  inline void SetDetailLevel(DetailLevel value) {
    m_detailLevelHasBeenSet = true;
    m_detailLevel = value;
  }
  inline ListAuditFindingsRequest& WithDetailLevel(DetailLevel value) {
    SetDetailLevel(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Include this value, if it was returned by the previous operation, to get the
   * next set of audit findings.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListAuditFindingsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of audit findings to return in one operation. If you omit
   * this parameter, the default of 10 is used.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListAuditFindingsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}
 private:
  Aws::Utils::DateTime m_startTime{};
  bool m_startTimeHasBeenSet = false;

  Aws::Utils::DateTime m_endTime{};
  bool m_endTimeHasBeenSet = false;

  Aws::Vector<Aws::String> m_auditors;
  bool m_auditorsHasBeenSet = false;

  Aws::Vector<AuditTarget> m_auditTargets;
  bool m_auditTargetsHasBeenSet = false;

  DetailLevel m_detailLevel{DetailLevel::NOT_SET};
  bool m_detailLevelHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;
};

}  // namespace Model
}  // namespace ApplicationSignals
}  // namespace Aws
