﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/gamelift/GameLiftRequest.h>
#include <aws/gamelift/GameLift_EXPORTS.h>
#include <aws/gamelift/model/Tag.h>

#include <utility>

namespace Aws {
namespace GameLift {
namespace Model {

/**
 */
class CreateMatchmakingRuleSetRequest : public GameLiftRequest {
 public:
  AWS_GAMELIFT_API CreateMatchmakingRuleSetRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateMatchmakingRuleSet"; }

  AWS_GAMELIFT_API Aws::String SerializePayload() const override;

  AWS_GAMELIFT_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>A unique identifier for the matchmaking rule set. A matchmaking configuration
   * identifies the rule set it uses by this name value. Note that the rule set name
   * is different from the optional <code>name</code> field in the rule set body.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateMatchmakingRuleSetRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A collection of matchmaking rules, formatted as a JSON string. Comments are
   * not allowed in JSON, but most elements support a description field.</p>
   */
  inline const Aws::String& GetRuleSetBody() const { return m_ruleSetBody; }
  inline bool RuleSetBodyHasBeenSet() const { return m_ruleSetBodyHasBeenSet; }
  template <typename RuleSetBodyT = Aws::String>
  void SetRuleSetBody(RuleSetBodyT&& value) {
    m_ruleSetBodyHasBeenSet = true;
    m_ruleSetBody = std::forward<RuleSetBodyT>(value);
  }
  template <typename RuleSetBodyT = Aws::String>
  CreateMatchmakingRuleSetRequest& WithRuleSetBody(RuleSetBodyT&& value) {
    SetRuleSetBody(std::forward<RuleSetBodyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of labels to assign to the new matchmaking rule set resource. Tags are
   * developer-defined key-value pairs. Tagging Amazon Web Services resources are
   * useful for resource management, access management and cost allocation. For more
   * information, see <a
   * href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html"> Tagging
   * Amazon Web Services Resources</a> in the <i>Amazon Web Services General
   * Reference</i>.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateMatchmakingRuleSetRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateMatchmakingRuleSetRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  Aws::String m_ruleSetBody;
  bool m_ruleSetBodyHasBeenSet = false;

  Aws::Vector<Tag> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace GameLift
}  // namespace Aws
