﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/greengrass/GreengrassRequest.h>
#include <aws/greengrass/Greengrass_EXPORTS.h>
#include <aws/greengrass/model/DeploymentType.h>

#include <utility>

namespace Aws {
namespace Greengrass {
namespace Model {

/**
 */
class CreateDeploymentRequest : public GreengrassRequest {
 public:
  AWS_GREENGRASS_API CreateDeploymentRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateDeployment"; }

  AWS_GREENGRASS_API Aws::String SerializePayload() const override;

  AWS_GREENGRASS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * A client token used to correlate requests and responses.
   */
  inline const Aws::String& GetAmznClientToken() const { return m_amznClientToken; }
  inline bool AmznClientTokenHasBeenSet() const { return m_amznClientTokenHasBeenSet; }
  template <typename AmznClientTokenT = Aws::String>
  void SetAmznClientToken(AmznClientTokenT&& value) {
    m_amznClientTokenHasBeenSet = true;
    m_amznClientToken = std::forward<AmznClientTokenT>(value);
  }
  template <typename AmznClientTokenT = Aws::String>
  CreateDeploymentRequest& WithAmznClientToken(AmznClientTokenT&& value) {
    SetAmznClientToken(std::forward<AmznClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * The ID of the deployment if you wish to redeploy a previous deployment.
   */
  inline const Aws::String& GetDeploymentId() const { return m_deploymentId; }
  inline bool DeploymentIdHasBeenSet() const { return m_deploymentIdHasBeenSet; }
  template <typename DeploymentIdT = Aws::String>
  void SetDeploymentId(DeploymentIdT&& value) {
    m_deploymentIdHasBeenSet = true;
    m_deploymentId = std::forward<DeploymentIdT>(value);
  }
  template <typename DeploymentIdT = Aws::String>
  CreateDeploymentRequest& WithDeploymentId(DeploymentIdT&& value) {
    SetDeploymentId(std::forward<DeploymentIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * The type of deployment. When used for ''CreateDeployment'', only
   * ''NewDeployment'' and ''Redeployment'' are valid.
   */
  inline DeploymentType GetDeploymentType() const { return m_deploymentType; }
  inline bool DeploymentTypeHasBeenSet() const { return m_deploymentTypeHasBeenSet; }
  inline void SetDeploymentType(DeploymentType value) {
    m_deploymentTypeHasBeenSet = true;
    m_deploymentType = value;
  }
  inline CreateDeploymentRequest& WithDeploymentType(DeploymentType value) {
    SetDeploymentType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * The ID of the Greengrass group.
   */
  inline const Aws::String& GetGroupId() const { return m_groupId; }
  inline bool GroupIdHasBeenSet() const { return m_groupIdHasBeenSet; }
  template <typename GroupIdT = Aws::String>
  void SetGroupId(GroupIdT&& value) {
    m_groupIdHasBeenSet = true;
    m_groupId = std::forward<GroupIdT>(value);
  }
  template <typename GroupIdT = Aws::String>
  CreateDeploymentRequest& WithGroupId(GroupIdT&& value) {
    SetGroupId(std::forward<GroupIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * The ID of the group version to be deployed.
   */
  inline const Aws::String& GetGroupVersionId() const { return m_groupVersionId; }
  inline bool GroupVersionIdHasBeenSet() const { return m_groupVersionIdHasBeenSet; }
  template <typename GroupVersionIdT = Aws::String>
  void SetGroupVersionId(GroupVersionIdT&& value) {
    m_groupVersionIdHasBeenSet = true;
    m_groupVersionId = std::forward<GroupVersionIdT>(value);
  }
  template <typename GroupVersionIdT = Aws::String>
  CreateDeploymentRequest& WithGroupVersionId(GroupVersionIdT&& value) {
    SetGroupVersionId(std::forward<GroupVersionIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_amznClientToken;
  bool m_amznClientTokenHasBeenSet = false;

  Aws::String m_deploymentId;
  bool m_deploymentIdHasBeenSet = false;

  DeploymentType m_deploymentType{DeploymentType::NOT_SET};
  bool m_deploymentTypeHasBeenSet = false;

  Aws::String m_groupId;
  bool m_groupIdHasBeenSet = false;

  Aws::String m_groupVersionId;
  bool m_groupVersionIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace Greengrass
}  // namespace Aws
