﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/kafka/KafkaRequest.h>
#include <aws/kafka/Kafka_EXPORTS.h>
#include <aws/kafka/model/ConnectivityInfo.h>

#include <utility>

namespace Aws {
namespace Kafka {
namespace Model {

/**
 * Request body for UpdateConnectivity.<p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/kafka-2018-11-14/UpdateConnectivityRequest">AWS
 * API Reference</a></p>
 */
class UpdateConnectivityRequest : public KafkaRequest {
 public:
  AWS_KAFKA_API UpdateConnectivityRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateConnectivity"; }

  AWS_KAFKA_API Aws::String SerializePayload() const override;

  ///@{
  /**
   *
          <p>The Amazon Resource Name (ARN) of the configuration.</p>

   *
   */
  inline const Aws::String& GetClusterArn() const { return m_clusterArn; }
  inline bool ClusterArnHasBeenSet() const { return m_clusterArnHasBeenSet; }
  template <typename ClusterArnT = Aws::String>
  void SetClusterArn(ClusterArnT&& value) {
    m_clusterArnHasBeenSet = true;
    m_clusterArn = std::forward<ClusterArnT>(value);
  }
  template <typename ClusterArnT = Aws::String>
  UpdateConnectivityRequest& WithClusterArn(ClusterArnT&& value) {
    SetClusterArn(std::forward<ClusterArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   *
          <p>Information about the broker access configuration.</p>

   */
  inline const ConnectivityInfo& GetConnectivityInfo() const { return m_connectivityInfo; }
  inline bool ConnectivityInfoHasBeenSet() const { return m_connectivityInfoHasBeenSet; }
  template <typename ConnectivityInfoT = ConnectivityInfo>
  void SetConnectivityInfo(ConnectivityInfoT&& value) {
    m_connectivityInfoHasBeenSet = true;
    m_connectivityInfo = std::forward<ConnectivityInfoT>(value);
  }
  template <typename ConnectivityInfoT = ConnectivityInfo>
  UpdateConnectivityRequest& WithConnectivityInfo(ConnectivityInfoT&& value) {
    SetConnectivityInfo(std::forward<ConnectivityInfoT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   *
          <p>The version of the MSK cluster to update. Cluster versions
   * aren't simple numbers. You can describe an MSK cluster to find its version. When
   * this update operation is successful, it generates a new cluster version.</p>

   *
   */
  inline const Aws::String& GetCurrentVersion() const { return m_currentVersion; }
  inline bool CurrentVersionHasBeenSet() const { return m_currentVersionHasBeenSet; }
  template <typename CurrentVersionT = Aws::String>
  void SetCurrentVersion(CurrentVersionT&& value) {
    m_currentVersionHasBeenSet = true;
    m_currentVersion = std::forward<CurrentVersionT>(value);
  }
  template <typename CurrentVersionT = Aws::String>
  UpdateConnectivityRequest& WithCurrentVersion(CurrentVersionT&& value) {
    SetCurrentVersion(std::forward<CurrentVersionT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_clusterArn;
  bool m_clusterArnHasBeenSet = false;

  ConnectivityInfo m_connectivityInfo;
  bool m_connectivityInfoHasBeenSet = false;

  Aws::String m_currentVersion;
  bool m_currentVersionHasBeenSet = false;
};

}  // namespace Model
}  // namespace Kafka
}  // namespace Aws
