﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/Array.h>
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/stream/ResponseStream.h>
#include <aws/mediastore-data/MediaStoreData_EXPORTS.h>

#include <utility>

namespace Aws {
template <typename RESULT_TYPE>
class AmazonWebServiceResult;

namespace MediaStoreData {
namespace Model {
class GetObjectResult {
 public:
  AWS_MEDIASTOREDATA_API GetObjectResult() = default;
  AWS_MEDIASTOREDATA_API GetObjectResult(GetObjectResult&&) = default;
  AWS_MEDIASTOREDATA_API GetObjectResult& operator=(GetObjectResult&&) = default;
  // we delete these because Microsoft doesn't handle move generation correctly
  // and we therefore don't trust them to get it right here either.
  GetObjectResult(const GetObjectResult&) = delete;
  GetObjectResult& operator=(const GetObjectResult&) = delete;

  AWS_MEDIASTOREDATA_API GetObjectResult(Aws::AmazonWebServiceResult<Aws::Utils::Stream::ResponseStream>&& result);
  AWS_MEDIASTOREDATA_API GetObjectResult& operator=(Aws::AmazonWebServiceResult<Aws::Utils::Stream::ResponseStream>&& result);

  ///@{
  /**
   * <p>The bytes of the object. </p>
   */
  inline Aws::IOStream& GetBody() const { return m_body.GetUnderlyingStream(); }
  inline void ReplaceBody(Aws::IOStream* body) { m_body = Aws::Utils::Stream::ResponseStream(body); }

  ///@}

  ///@{
  /**
   * <p>An optional <code>CacheControl</code> header that allows the caller to
   * control the object's cache behavior. Headers can be passed in as specified in
   * the HTTP spec at <a
   * href="https://www.w3.org/Protocols/rfc2616/rfc2616-sec14.html#sec14.9">https://www.w3.org/Protocols/rfc2616/rfc2616-sec14.html#sec14.9</a>.</p>
   * <p>Headers with a custom user-defined value are also accepted.</p>
   */
  inline const Aws::String& GetCacheControl() const { return m_cacheControl; }
  template <typename CacheControlT = Aws::String>
  void SetCacheControl(CacheControlT&& value) {
    m_cacheControlHasBeenSet = true;
    m_cacheControl = std::forward<CacheControlT>(value);
  }
  template <typename CacheControlT = Aws::String>
  GetObjectResult& WithCacheControl(CacheControlT&& value) {
    SetCacheControl(std::forward<CacheControlT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The range of bytes to retrieve.</p>
   */
  inline const Aws::String& GetContentRange() const { return m_contentRange; }
  template <typename ContentRangeT = Aws::String>
  void SetContentRange(ContentRangeT&& value) {
    m_contentRangeHasBeenSet = true;
    m_contentRange = std::forward<ContentRangeT>(value);
  }
  template <typename ContentRangeT = Aws::String>
  GetObjectResult& WithContentRange(ContentRangeT&& value) {
    SetContentRange(std::forward<ContentRangeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The length of the object in bytes.</p>
   */
  inline long long GetContentLength() const { return m_contentLength; }
  inline void SetContentLength(long long value) {
    m_contentLengthHasBeenSet = true;
    m_contentLength = value;
  }
  inline GetObjectResult& WithContentLength(long long value) {
    SetContentLength(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The content type of the object.</p>
   */
  inline const Aws::String& GetContentType() const { return m_contentType; }
  template <typename ContentTypeT = Aws::String>
  void SetContentType(ContentTypeT&& value) {
    m_contentTypeHasBeenSet = true;
    m_contentType = std::forward<ContentTypeT>(value);
  }
  template <typename ContentTypeT = Aws::String>
  GetObjectResult& WithContentType(ContentTypeT&& value) {
    SetContentType(std::forward<ContentTypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ETag that represents a unique instance of the object.</p>
   */
  inline const Aws::String& GetETag() const { return m_eTag; }
  template <typename ETagT = Aws::String>
  void SetETag(ETagT&& value) {
    m_eTagHasBeenSet = true;
    m_eTag = std::forward<ETagT>(value);
  }
  template <typename ETagT = Aws::String>
  GetObjectResult& WithETag(ETagT&& value) {
    SetETag(std::forward<ETagT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time that the object was last modified.</p>
   */
  inline const Aws::Utils::DateTime& GetLastModified() const { return m_lastModified; }
  template <typename LastModifiedT = Aws::Utils::DateTime>
  void SetLastModified(LastModifiedT&& value) {
    m_lastModifiedHasBeenSet = true;
    m_lastModified = std::forward<LastModifiedT>(value);
  }
  template <typename LastModifiedT = Aws::Utils::DateTime>
  GetObjectResult& WithLastModified(LastModifiedT&& value) {
    SetLastModified(std::forward<LastModifiedT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The HTML status code of the request. Status codes ranging from 200 to 299
   * indicate success. All other status codes indicate the type of error that
   * occurred.</p>
   */
  inline int GetStatusCode() const { return m_statusCode; }
  inline void SetStatusCode(int value) {
    m_statusCodeHasBeenSet = true;
    m_statusCode = value;
  }
  inline GetObjectResult& WithStatusCode(int value) {
    SetStatusCode(value);
    return *this;
  }
  ///@}

  ///@{

  inline const Aws::String& GetRequestId() const { return m_requestId; }
  template <typename RequestIdT = Aws::String>
  void SetRequestId(RequestIdT&& value) {
    m_requestIdHasBeenSet = true;
    m_requestId = std::forward<RequestIdT>(value);
  }
  template <typename RequestIdT = Aws::String>
  GetObjectResult& WithRequestId(RequestIdT&& value) {
    SetRequestId(std::forward<RequestIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::Utils::Stream::ResponseStream m_body{};
  bool m_bodyHasBeenSet = false;

  Aws::String m_cacheControl;
  bool m_cacheControlHasBeenSet = false;

  Aws::String m_contentRange;
  bool m_contentRangeHasBeenSet = false;

  long long m_contentLength{0};
  bool m_contentLengthHasBeenSet = false;

  Aws::String m_contentType;
  bool m_contentTypeHasBeenSet = false;

  Aws::String m_eTag;
  bool m_eTagHasBeenSet = false;

  Aws::Utils::DateTime m_lastModified{};
  bool m_lastModifiedHasBeenSet = false;

  int m_statusCode{0};
  bool m_statusCodeHasBeenSet = false;

  Aws::String m_requestId;
  bool m_requestIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace MediaStoreData
}  // namespace Aws
