﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSStreamFwd.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/rds/RDS_EXPORTS.h>
#include <aws/rds/model/AdditionalStorageVolume.h>
#include <aws/rds/model/AutomationMode.h>
#include <aws/rds/model/PendingCloudwatchLogsExports.h>
#include <aws/rds/model/ProcessorFeature.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace RDS {
namespace Model {

/**
 * <p>This data type is used as a response element in the
 * <code>ModifyDBInstance</code> operation and contains changes that will be
 * applied during the next maintenance window.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/rds-2014-10-31/PendingModifiedValues">AWS
 * API Reference</a></p>
 */
class PendingModifiedValues {
 public:
  AWS_RDS_API PendingModifiedValues() = default;
  AWS_RDS_API PendingModifiedValues(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_RDS_API PendingModifiedValues& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_RDS_API void OutputToStream(Aws::OStream& ostream, const char* location, unsigned index, const char* locationValue) const;
  AWS_RDS_API void OutputToStream(Aws::OStream& oStream, const char* location) const;

  ///@{
  /**
   * <p>The name of the compute and memory capacity class for the DB instance.</p>
   */
  inline const Aws::String& GetDBInstanceClass() const { return m_dBInstanceClass; }
  inline bool DBInstanceClassHasBeenSet() const { return m_dBInstanceClassHasBeenSet; }
  template <typename DBInstanceClassT = Aws::String>
  void SetDBInstanceClass(DBInstanceClassT&& value) {
    m_dBInstanceClassHasBeenSet = true;
    m_dBInstanceClass = std::forward<DBInstanceClassT>(value);
  }
  template <typename DBInstanceClassT = Aws::String>
  PendingModifiedValues& WithDBInstanceClass(DBInstanceClassT&& value) {
    SetDBInstanceClass(std::forward<DBInstanceClassT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The allocated storage size for the DB instance specified in gibibytes
   * (GiB).</p>
   */
  inline int GetAllocatedStorage() const { return m_allocatedStorage; }
  inline bool AllocatedStorageHasBeenSet() const { return m_allocatedStorageHasBeenSet; }
  inline void SetAllocatedStorage(int value) {
    m_allocatedStorageHasBeenSet = true;
    m_allocatedStorage = value;
  }
  inline PendingModifiedValues& WithAllocatedStorage(int value) {
    SetAllocatedStorage(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The master credentials for the DB instance.</p>
   */
  inline const Aws::String& GetMasterUserPassword() const { return m_masterUserPassword; }
  inline bool MasterUserPasswordHasBeenSet() const { return m_masterUserPasswordHasBeenSet; }
  template <typename MasterUserPasswordT = Aws::String>
  void SetMasterUserPassword(MasterUserPasswordT&& value) {
    m_masterUserPasswordHasBeenSet = true;
    m_masterUserPassword = std::forward<MasterUserPasswordT>(value);
  }
  template <typename MasterUserPasswordT = Aws::String>
  PendingModifiedValues& WithMasterUserPassword(MasterUserPasswordT&& value) {
    SetMasterUserPassword(std::forward<MasterUserPasswordT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The port for the DB instance.</p>
   */
  inline int GetPort() const { return m_port; }
  inline bool PortHasBeenSet() const { return m_portHasBeenSet; }
  inline void SetPort(int value) {
    m_portHasBeenSet = true;
    m_port = value;
  }
  inline PendingModifiedValues& WithPort(int value) {
    SetPort(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The number of days for which automated backups are retained.</p>
   */
  inline int GetBackupRetentionPeriod() const { return m_backupRetentionPeriod; }
  inline bool BackupRetentionPeriodHasBeenSet() const { return m_backupRetentionPeriodHasBeenSet; }
  inline void SetBackupRetentionPeriod(int value) {
    m_backupRetentionPeriodHasBeenSet = true;
    m_backupRetentionPeriod = value;
  }
  inline PendingModifiedValues& WithBackupRetentionPeriod(int value) {
    SetBackupRetentionPeriod(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates whether the Single-AZ DB instance will change to a Multi-AZ
   * deployment.</p>
   */
  inline bool GetMultiAZ() const { return m_multiAZ; }
  inline bool MultiAZHasBeenSet() const { return m_multiAZHasBeenSet; }
  inline void SetMultiAZ(bool value) {
    m_multiAZHasBeenSet = true;
    m_multiAZ = value;
  }
  inline PendingModifiedValues& WithMultiAZ(bool value) {
    SetMultiAZ(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The database engine version.</p>
   */
  inline const Aws::String& GetEngineVersion() const { return m_engineVersion; }
  inline bool EngineVersionHasBeenSet() const { return m_engineVersionHasBeenSet; }
  template <typename EngineVersionT = Aws::String>
  void SetEngineVersion(EngineVersionT&& value) {
    m_engineVersionHasBeenSet = true;
    m_engineVersion = std::forward<EngineVersionT>(value);
  }
  template <typename EngineVersionT = Aws::String>
  PendingModifiedValues& WithEngineVersion(EngineVersionT&& value) {
    SetEngineVersion(std::forward<EngineVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The license model for the DB instance.</p> <p>Valid values:
   * <code>license-included</code> | <code>bring-your-own-license</code> |
   * <code>general-public-license</code> </p>
   */
  inline const Aws::String& GetLicenseModel() const { return m_licenseModel; }
  inline bool LicenseModelHasBeenSet() const { return m_licenseModelHasBeenSet; }
  template <typename LicenseModelT = Aws::String>
  void SetLicenseModel(LicenseModelT&& value) {
    m_licenseModelHasBeenSet = true;
    m_licenseModel = std::forward<LicenseModelT>(value);
  }
  template <typename LicenseModelT = Aws::String>
  PendingModifiedValues& WithLicenseModel(LicenseModelT&& value) {
    SetLicenseModel(std::forward<LicenseModelT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Provisioned IOPS value for the DB instance.</p>
   */
  inline int GetIops() const { return m_iops; }
  inline bool IopsHasBeenSet() const { return m_iopsHasBeenSet; }
  inline void SetIops(int value) {
    m_iopsHasBeenSet = true;
    m_iops = value;
  }
  inline PendingModifiedValues& WithIops(int value) {
    SetIops(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The storage throughput of the DB instance.</p>
   */
  inline int GetStorageThroughput() const { return m_storageThroughput; }
  inline bool StorageThroughputHasBeenSet() const { return m_storageThroughputHasBeenSet; }
  inline void SetStorageThroughput(int value) {
    m_storageThroughputHasBeenSet = true;
    m_storageThroughput = value;
  }
  inline PendingModifiedValues& WithStorageThroughput(int value) {
    SetStorageThroughput(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The database identifier for the DB instance.</p>
   */
  inline const Aws::String& GetDBInstanceIdentifier() const { return m_dBInstanceIdentifier; }
  inline bool DBInstanceIdentifierHasBeenSet() const { return m_dBInstanceIdentifierHasBeenSet; }
  template <typename DBInstanceIdentifierT = Aws::String>
  void SetDBInstanceIdentifier(DBInstanceIdentifierT&& value) {
    m_dBInstanceIdentifierHasBeenSet = true;
    m_dBInstanceIdentifier = std::forward<DBInstanceIdentifierT>(value);
  }
  template <typename DBInstanceIdentifierT = Aws::String>
  PendingModifiedValues& WithDBInstanceIdentifier(DBInstanceIdentifierT&& value) {
    SetDBInstanceIdentifier(std::forward<DBInstanceIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The storage type of the DB instance.</p>
   */
  inline const Aws::String& GetStorageType() const { return m_storageType; }
  inline bool StorageTypeHasBeenSet() const { return m_storageTypeHasBeenSet; }
  template <typename StorageTypeT = Aws::String>
  void SetStorageType(StorageTypeT&& value) {
    m_storageTypeHasBeenSet = true;
    m_storageType = std::forward<StorageTypeT>(value);
  }
  template <typename StorageTypeT = Aws::String>
  PendingModifiedValues& WithStorageType(StorageTypeT&& value) {
    SetStorageType(std::forward<StorageTypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the CA certificate for the DB instance.</p> <p>For more
   * information, see <a
   * href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.SSL.html">Using
   * SSL/TLS to encrypt a connection to a DB instance</a> in the <i>Amazon RDS User
   * Guide</i> and <a
   * href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/UsingWithRDS.SSL.html">
   * Using SSL/TLS to encrypt a connection to a DB cluster</a> in the <i>Amazon
   * Aurora User Guide</i>.</p>
   */
  inline const Aws::String& GetCACertificateIdentifier() const { return m_cACertificateIdentifier; }
  inline bool CACertificateIdentifierHasBeenSet() const { return m_cACertificateIdentifierHasBeenSet; }
  template <typename CACertificateIdentifierT = Aws::String>
  void SetCACertificateIdentifier(CACertificateIdentifierT&& value) {
    m_cACertificateIdentifierHasBeenSet = true;
    m_cACertificateIdentifier = std::forward<CACertificateIdentifierT>(value);
  }
  template <typename CACertificateIdentifierT = Aws::String>
  PendingModifiedValues& WithCACertificateIdentifier(CACertificateIdentifierT&& value) {
    SetCACertificateIdentifier(std::forward<CACertificateIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The DB subnet group for the DB instance.</p>
   */
  inline const Aws::String& GetDBSubnetGroupName() const { return m_dBSubnetGroupName; }
  inline bool DBSubnetGroupNameHasBeenSet() const { return m_dBSubnetGroupNameHasBeenSet; }
  template <typename DBSubnetGroupNameT = Aws::String>
  void SetDBSubnetGroupName(DBSubnetGroupNameT&& value) {
    m_dBSubnetGroupNameHasBeenSet = true;
    m_dBSubnetGroupName = std::forward<DBSubnetGroupNameT>(value);
  }
  template <typename DBSubnetGroupNameT = Aws::String>
  PendingModifiedValues& WithDBSubnetGroupName(DBSubnetGroupNameT&& value) {
    SetDBSubnetGroupName(std::forward<DBSubnetGroupNameT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const PendingCloudwatchLogsExports& GetPendingCloudwatchLogsExports() const { return m_pendingCloudwatchLogsExports; }
  inline bool PendingCloudwatchLogsExportsHasBeenSet() const { return m_pendingCloudwatchLogsExportsHasBeenSet; }
  template <typename PendingCloudwatchLogsExportsT = PendingCloudwatchLogsExports>
  void SetPendingCloudwatchLogsExports(PendingCloudwatchLogsExportsT&& value) {
    m_pendingCloudwatchLogsExportsHasBeenSet = true;
    m_pendingCloudwatchLogsExports = std::forward<PendingCloudwatchLogsExportsT>(value);
  }
  template <typename PendingCloudwatchLogsExportsT = PendingCloudwatchLogsExports>
  PendingModifiedValues& WithPendingCloudwatchLogsExports(PendingCloudwatchLogsExportsT&& value) {
    SetPendingCloudwatchLogsExports(std::forward<PendingCloudwatchLogsExportsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The number of CPU cores and the number of threads per core for the DB
   * instance class of the DB instance.</p>
   */
  inline const Aws::Vector<ProcessorFeature>& GetProcessorFeatures() const { return m_processorFeatures; }
  inline bool ProcessorFeaturesHasBeenSet() const { return m_processorFeaturesHasBeenSet; }
  template <typename ProcessorFeaturesT = Aws::Vector<ProcessorFeature>>
  void SetProcessorFeatures(ProcessorFeaturesT&& value) {
    m_processorFeaturesHasBeenSet = true;
    m_processorFeatures = std::forward<ProcessorFeaturesT>(value);
  }
  template <typename ProcessorFeaturesT = Aws::Vector<ProcessorFeature>>
  PendingModifiedValues& WithProcessorFeatures(ProcessorFeaturesT&& value) {
    SetProcessorFeatures(std::forward<ProcessorFeaturesT>(value));
    return *this;
  }
  template <typename ProcessorFeaturesT = ProcessorFeature>
  PendingModifiedValues& AddProcessorFeatures(ProcessorFeaturesT&& value) {
    m_processorFeaturesHasBeenSet = true;
    m_processorFeatures.emplace_back(std::forward<ProcessorFeaturesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The automation mode of the RDS Custom DB instance: <code>full</code> or
   * <code>all-paused</code>. If <code>full</code>, the DB instance automates
   * monitoring and instance recovery. If <code>all-paused</code>, the instance
   * pauses automation for the duration set by
   * <code>--resume-full-automation-mode-minutes</code>.</p>
   */
  inline AutomationMode GetAutomationMode() const { return m_automationMode; }
  inline bool AutomationModeHasBeenSet() const { return m_automationModeHasBeenSet; }
  inline void SetAutomationMode(AutomationMode value) {
    m_automationModeHasBeenSet = true;
    m_automationMode = value;
  }
  inline PendingModifiedValues& WithAutomationMode(AutomationMode value) {
    SetAutomationMode(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The number of minutes to pause the automation. When the time period ends, RDS
   * Custom resumes full automation. The minimum value is 60 (default). The maximum
   * value is 1,440.</p>
   */
  inline const Aws::Utils::DateTime& GetResumeFullAutomationModeTime() const { return m_resumeFullAutomationModeTime; }
  inline bool ResumeFullAutomationModeTimeHasBeenSet() const { return m_resumeFullAutomationModeTimeHasBeenSet; }
  template <typename ResumeFullAutomationModeTimeT = Aws::Utils::DateTime>
  void SetResumeFullAutomationModeTime(ResumeFullAutomationModeTimeT&& value) {
    m_resumeFullAutomationModeTimeHasBeenSet = true;
    m_resumeFullAutomationModeTime = std::forward<ResumeFullAutomationModeTimeT>(value);
  }
  template <typename ResumeFullAutomationModeTimeT = Aws::Utils::DateTime>
  PendingModifiedValues& WithResumeFullAutomationModeTime(ResumeFullAutomationModeTimeT&& value) {
    SetResumeFullAutomationModeTime(std::forward<ResumeFullAutomationModeTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates whether the DB instance will change to the multi-tenant
   * configuration (TRUE) or the single-tenant configuration (FALSE). </p>
   */
  inline bool GetMultiTenant() const { return m_multiTenant; }
  inline bool MultiTenantHasBeenSet() const { return m_multiTenantHasBeenSet; }
  inline void SetMultiTenant(bool value) {
    m_multiTenantHasBeenSet = true;
    m_multiTenant = value;
  }
  inline PendingModifiedValues& WithMultiTenant(bool value) {
    SetMultiTenant(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates whether mapping of Amazon Web Services Identity and Access
   * Management (IAM) accounts to database accounts is enabled.</p>
   */
  inline bool GetIAMDatabaseAuthenticationEnabled() const { return m_iAMDatabaseAuthenticationEnabled; }
  inline bool IAMDatabaseAuthenticationEnabledHasBeenSet() const { return m_iAMDatabaseAuthenticationEnabledHasBeenSet; }
  inline void SetIAMDatabaseAuthenticationEnabled(bool value) {
    m_iAMDatabaseAuthenticationEnabledHasBeenSet = true;
    m_iAMDatabaseAuthenticationEnabled = value;
  }
  inline PendingModifiedValues& WithIAMDatabaseAuthenticationEnabled(bool value) {
    SetIAMDatabaseAuthenticationEnabled(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates whether the DB instance has a dedicated log volume (DLV)
   * enabled.&gt;</p>
   */
  inline bool GetDedicatedLogVolume() const { return m_dedicatedLogVolume; }
  inline bool DedicatedLogVolumeHasBeenSet() const { return m_dedicatedLogVolumeHasBeenSet; }
  inline void SetDedicatedLogVolume(bool value) {
    m_dedicatedLogVolumeHasBeenSet = true;
    m_dedicatedLogVolume = value;
  }
  inline PendingModifiedValues& WithDedicatedLogVolume(bool value) {
    SetDedicatedLogVolume(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The database engine of the DB instance.</p>
   */
  inline const Aws::String& GetEngine() const { return m_engine; }
  inline bool EngineHasBeenSet() const { return m_engineHasBeenSet; }
  template <typename EngineT = Aws::String>
  void SetEngine(EngineT&& value) {
    m_engineHasBeenSet = true;
    m_engine = std::forward<EngineT>(value);
  }
  template <typename EngineT = Aws::String>
  PendingModifiedValues& WithEngine(EngineT&& value) {
    SetEngine(std::forward<EngineT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The additional storage volume modifications that are pending for the DB
   * instance.</p>
   */
  inline const Aws::Vector<AdditionalStorageVolume>& GetAdditionalStorageVolumes() const { return m_additionalStorageVolumes; }
  inline bool AdditionalStorageVolumesHasBeenSet() const { return m_additionalStorageVolumesHasBeenSet; }
  template <typename AdditionalStorageVolumesT = Aws::Vector<AdditionalStorageVolume>>
  void SetAdditionalStorageVolumes(AdditionalStorageVolumesT&& value) {
    m_additionalStorageVolumesHasBeenSet = true;
    m_additionalStorageVolumes = std::forward<AdditionalStorageVolumesT>(value);
  }
  template <typename AdditionalStorageVolumesT = Aws::Vector<AdditionalStorageVolume>>
  PendingModifiedValues& WithAdditionalStorageVolumes(AdditionalStorageVolumesT&& value) {
    SetAdditionalStorageVolumes(std::forward<AdditionalStorageVolumesT>(value));
    return *this;
  }
  template <typename AdditionalStorageVolumesT = AdditionalStorageVolume>
  PendingModifiedValues& AddAdditionalStorageVolumes(AdditionalStorageVolumesT&& value) {
    m_additionalStorageVolumesHasBeenSet = true;
    m_additionalStorageVolumes.emplace_back(std::forward<AdditionalStorageVolumesT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_dBInstanceClass;
  bool m_dBInstanceClassHasBeenSet = false;

  int m_allocatedStorage{0};
  bool m_allocatedStorageHasBeenSet = false;

  Aws::String m_masterUserPassword;
  bool m_masterUserPasswordHasBeenSet = false;

  int m_port{0};
  bool m_portHasBeenSet = false;

  int m_backupRetentionPeriod{0};
  bool m_backupRetentionPeriodHasBeenSet = false;

  bool m_multiAZ{false};
  bool m_multiAZHasBeenSet = false;

  Aws::String m_engineVersion;
  bool m_engineVersionHasBeenSet = false;

  Aws::String m_licenseModel;
  bool m_licenseModelHasBeenSet = false;

  int m_iops{0};
  bool m_iopsHasBeenSet = false;

  int m_storageThroughput{0};
  bool m_storageThroughputHasBeenSet = false;

  Aws::String m_dBInstanceIdentifier;
  bool m_dBInstanceIdentifierHasBeenSet = false;

  Aws::String m_storageType;
  bool m_storageTypeHasBeenSet = false;

  Aws::String m_cACertificateIdentifier;
  bool m_cACertificateIdentifierHasBeenSet = false;

  Aws::String m_dBSubnetGroupName;
  bool m_dBSubnetGroupNameHasBeenSet = false;

  PendingCloudwatchLogsExports m_pendingCloudwatchLogsExports;
  bool m_pendingCloudwatchLogsExportsHasBeenSet = false;

  Aws::Vector<ProcessorFeature> m_processorFeatures;
  bool m_processorFeaturesHasBeenSet = false;

  AutomationMode m_automationMode{AutomationMode::NOT_SET};
  bool m_automationModeHasBeenSet = false;

  Aws::Utils::DateTime m_resumeFullAutomationModeTime{};
  bool m_resumeFullAutomationModeTimeHasBeenSet = false;

  bool m_multiTenant{false};
  bool m_multiTenantHasBeenSet = false;

  bool m_iAMDatabaseAuthenticationEnabled{false};
  bool m_iAMDatabaseAuthenticationEnabledHasBeenSet = false;

  bool m_dedicatedLogVolume{false};
  bool m_dedicatedLogVolumeHasBeenSet = false;

  Aws::String m_engine;
  bool m_engineHasBeenSet = false;

  Aws::Vector<AdditionalStorageVolume> m_additionalStorageVolumes;
  bool m_additionalStorageVolumesHasBeenSet = false;
};

}  // namespace Model
}  // namespace RDS
}  // namespace Aws
