﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ssm-contacts/SSMContactsRequest.h>
#include <aws/ssm-contacts/SSMContacts_EXPORTS.h>
#include <aws/ssm-contacts/model/PreviewOverride.h>
#include <aws/ssm-contacts/model/RecurrenceSettings.h>

#include <utility>

namespace Aws {
namespace SSMContacts {
namespace Model {

/**
 */
class ListPreviewRotationShiftsRequest : public SSMContactsRequest {
 public:
  AWS_SSMCONTACTS_API ListPreviewRotationShiftsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListPreviewRotationShifts"; }

  AWS_SSMCONTACTS_API Aws::String SerializePayload() const override;

  AWS_SSMCONTACTS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The date and time a rotation would begin. The first shift is calculated from
   * this date and time.</p>
   */
  inline const Aws::Utils::DateTime& GetRotationStartTime() const { return m_rotationStartTime; }
  inline bool RotationStartTimeHasBeenSet() const { return m_rotationStartTimeHasBeenSet; }
  template <typename RotationStartTimeT = Aws::Utils::DateTime>
  void SetRotationStartTime(RotationStartTimeT&& value) {
    m_rotationStartTimeHasBeenSet = true;
    m_rotationStartTime = std::forward<RotationStartTimeT>(value);
  }
  template <typename RotationStartTimeT = Aws::Utils::DateTime>
  ListPreviewRotationShiftsRequest& WithRotationStartTime(RotationStartTimeT&& value) {
    SetRotationStartTime(std::forward<RotationStartTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Used to filter the range of calculated shifts before sending the response
   * back to the user. </p>
   */
  inline const Aws::Utils::DateTime& GetStartTime() const { return m_startTime; }
  inline bool StartTimeHasBeenSet() const { return m_startTimeHasBeenSet; }
  template <typename StartTimeT = Aws::Utils::DateTime>
  void SetStartTime(StartTimeT&& value) {
    m_startTimeHasBeenSet = true;
    m_startTime = std::forward<StartTimeT>(value);
  }
  template <typename StartTimeT = Aws::Utils::DateTime>
  ListPreviewRotationShiftsRequest& WithStartTime(StartTimeT&& value) {
    SetStartTime(std::forward<StartTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time a rotation shift would end.</p>
   */
  inline const Aws::Utils::DateTime& GetEndTime() const { return m_endTime; }
  inline bool EndTimeHasBeenSet() const { return m_endTimeHasBeenSet; }
  template <typename EndTimeT = Aws::Utils::DateTime>
  void SetEndTime(EndTimeT&& value) {
    m_endTimeHasBeenSet = true;
    m_endTime = std::forward<EndTimeT>(value);
  }
  template <typename EndTimeT = Aws::Utils::DateTime>
  ListPreviewRotationShiftsRequest& WithEndTime(EndTimeT&& value) {
    SetEndTime(std::forward<EndTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The contacts that would be assigned to a rotation.</p>
   */
  inline const Aws::Vector<Aws::String>& GetMembers() const { return m_members; }
  inline bool MembersHasBeenSet() const { return m_membersHasBeenSet; }
  template <typename MembersT = Aws::Vector<Aws::String>>
  void SetMembers(MembersT&& value) {
    m_membersHasBeenSet = true;
    m_members = std::forward<MembersT>(value);
  }
  template <typename MembersT = Aws::Vector<Aws::String>>
  ListPreviewRotationShiftsRequest& WithMembers(MembersT&& value) {
    SetMembers(std::forward<MembersT>(value));
    return *this;
  }
  template <typename MembersT = Aws::String>
  ListPreviewRotationShiftsRequest& AddMembers(MembersT&& value) {
    m_membersHasBeenSet = true;
    m_members.emplace_back(std::forward<MembersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The time zone the rotation’s activity would be based on, in Internet Assigned
   * Numbers Authority (IANA) format. For example: "America/Los_Angeles", "UTC", or
   * "Asia/Seoul". </p>
   */
  inline const Aws::String& GetTimeZoneId() const { return m_timeZoneId; }
  inline bool TimeZoneIdHasBeenSet() const { return m_timeZoneIdHasBeenSet; }
  template <typename TimeZoneIdT = Aws::String>
  void SetTimeZoneId(TimeZoneIdT&& value) {
    m_timeZoneIdHasBeenSet = true;
    m_timeZoneId = std::forward<TimeZoneIdT>(value);
  }
  template <typename TimeZoneIdT = Aws::String>
  ListPreviewRotationShiftsRequest& WithTimeZoneId(TimeZoneIdT&& value) {
    SetTimeZoneId(std::forward<TimeZoneIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Information about how long a rotation would last before restarting at the
   * beginning of the shift order.</p>
   */
  inline const RecurrenceSettings& GetRecurrence() const { return m_recurrence; }
  inline bool RecurrenceHasBeenSet() const { return m_recurrenceHasBeenSet; }
  template <typename RecurrenceT = RecurrenceSettings>
  void SetRecurrence(RecurrenceT&& value) {
    m_recurrenceHasBeenSet = true;
    m_recurrence = std::forward<RecurrenceT>(value);
  }
  template <typename RecurrenceT = RecurrenceSettings>
  ListPreviewRotationShiftsRequest& WithRecurrence(RecurrenceT&& value) {
    SetRecurrence(std::forward<RecurrenceT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Information about changes that would be made in a rotation override.</p>
   */
  inline const Aws::Vector<PreviewOverride>& GetOverrides() const { return m_overrides; }
  inline bool OverridesHasBeenSet() const { return m_overridesHasBeenSet; }
  template <typename OverridesT = Aws::Vector<PreviewOverride>>
  void SetOverrides(OverridesT&& value) {
    m_overridesHasBeenSet = true;
    m_overrides = std::forward<OverridesT>(value);
  }
  template <typename OverridesT = Aws::Vector<PreviewOverride>>
  ListPreviewRotationShiftsRequest& WithOverrides(OverridesT&& value) {
    SetOverrides(std::forward<OverridesT>(value));
    return *this;
  }
  template <typename OverridesT = PreviewOverride>
  ListPreviewRotationShiftsRequest& AddOverrides(OverridesT&& value) {
    m_overridesHasBeenSet = true;
    m_overrides.emplace_back(std::forward<OverridesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A token to start the list. This token is used to get the next set of
   * results.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListPreviewRotationShiftsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of items to return for this call. The call also returns a
   * token that can be specified in a subsequent call to get the next set of
   * results.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListPreviewRotationShiftsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}
 private:
  Aws::Utils::DateTime m_rotationStartTime{};
  bool m_rotationStartTimeHasBeenSet = false;

  Aws::Utils::DateTime m_startTime{};
  bool m_startTimeHasBeenSet = false;

  Aws::Utils::DateTime m_endTime{};
  bool m_endTimeHasBeenSet = false;

  Aws::Vector<Aws::String> m_members;
  bool m_membersHasBeenSet = false;

  Aws::String m_timeZoneId;
  bool m_timeZoneIdHasBeenSet = false;

  RecurrenceSettings m_recurrence;
  bool m_recurrenceHasBeenSet = false;

  Aws::Vector<PreviewOverride> m_overrides;
  bool m_overridesHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;
};

}  // namespace Model
}  // namespace SSMContacts
}  // namespace Aws
