/*
 * Decompiled with CFR 0.152.
 */
package utilities.util.reflection;

import ghidra.util.Msg;
import ghidra.util.exception.AssertException;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.lang.reflect.Array;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.GenericArrayType;
import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.lang.reflect.TypeVariable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

public class ReflectionUtilities {
    private static final String JAVA_AWT_PATTERN = "java.awt";
    private static final String JAVA_REFLECT_PATTERN = "java.lang.reflect";
    private static final String JDK_INTERNAL_REFLECT_PATTERN = "jdk.internal.reflect";
    private static final String SWING_JAVA_PATTERN = "java.swing";
    private static final String SWING_JAVAX_PATTERN = "javax.swing";
    private static final String SUN_AWT_PATTERN = "sun.awt";
    private static final String SUN_REFLECT_PATTERN = "sun.reflect";
    private static final String SECURITY_PATTERN = "java.security";
    private static final String JUNIT_PATTERN = ".junit";
    private static final String MOCKIT_PATTERN = "mockit";

    private ReflectionUtilities() {
    }

    public static Field locateStaticFieldObjectOnClass(String fieldName, Class<?> containingClass) {
        Field field;
        block2: {
            field = null;
            try {
                field = containingClass.getDeclaredField(fieldName);
            }
            catch (NoSuchFieldException nsfe) {
                Class<?> parentClass = containingClass.getSuperclass();
                if (parentClass == null) break block2;
                field = ReflectionUtilities.locateFieldObjectOnClass(fieldName, parentClass);
            }
        }
        return field;
    }

    public static Field locateFieldObjectOnClass(String fieldName, Class<?> containingClass) {
        Field field;
        block2: {
            field = null;
            try {
                field = containingClass.getDeclaredField(fieldName);
            }
            catch (NoSuchFieldException nsfe) {
                Class<?> parentClass = containingClass.getSuperclass();
                if (parentClass == null) break block2;
                field = ReflectionUtilities.locateFieldObjectOnClass(fieldName, parentClass);
            }
        }
        return field;
    }

    public static Method locateMethodObjectOnClass(String methodName, Class<?> containingClass, Class<?>[] parameterTypes) {
        Method method;
        block2: {
            method = null;
            try {
                method = containingClass.getDeclaredMethod(methodName, parameterTypes);
            }
            catch (NoSuchMethodException nsme) {
                Class<?> parentClass = containingClass.getSuperclass();
                if (parentClass == null) break block2;
                method = ReflectionUtilities.locateMethodObjectOnClass(methodName, parentClass, parameterTypes);
            }
        }
        return method;
    }

    public static Constructor<?> locateConstructorOnClass(Class<?> containingClass, Class<?>[] parameterTypes) {
        Constructor<?> constructor = null;
        try {
            constructor = containingClass.getDeclaredConstructor(parameterTypes);
        }
        catch (SecurityException securityException) {
        }
        catch (NoSuchMethodException noSuchMethodException) {
            // empty catch block
        }
        return constructor;
    }

    public static Field locateFieldByTypeOnClass(Class<?> classType, Class<?> containingClass) {
        Field[] declaredFields;
        for (Field field : declaredFields = containingClass.getDeclaredFields()) {
            Class<?> fieldClass = field.getType();
            if (fieldClass != classType) continue;
            return field;
        }
        Class<?> parentClass = containingClass.getSuperclass();
        if (parentClass == null) {
            return null;
        }
        return ReflectionUtilities.locateFieldByTypeOnClass(classType, parentClass);
    }

    public static String getClassNameOlderThan(Class<?> ... classes) {
        Throwable t = ReflectionUtilities.createThrowableWithStackOlderThan(classes);
        StackTraceElement[] stackTrace = t.getStackTrace();
        return stackTrace[0].getClassName();
    }

    public static String getClassNameOlderThan(String ... patterns) {
        Throwable t = ReflectionUtilities.createThrowableWithStackOlderThan(patterns);
        StackTraceElement[] stackTrace = t.getStackTrace();
        return stackTrace[0].getClassName();
    }

    public static Throwable createThrowableWithStackOlderThan(String ... patterns) {
        return ReflectionUtilities.createThrowableWithStackOlderThan(List.of(patterns));
    }

    private static Throwable createThrowableWithStackOlderThan(List<String> patterns) {
        return ReflectionUtilities.createThrowableWithStackOlderThan(patterns, StackElementMatcher.CONTAINS_CLASS_OR_METHOD);
    }

    private static Throwable createThrowableWithStackOlderThan(List<String> patterns, StackElementMatcher matcher) {
        if (patterns.isEmpty()) {
            patterns = new ArrayList<String>();
            patterns.add(ReflectionUtilities.class.getName());
        }
        Throwable t = new Throwable();
        StackTraceElement[] trace = t.getStackTrace();
        int lastIgnoreIndex = -1;
        for (int i = 0; i < trace.length; ++i) {
            StackTraceElement element = trace[i];
            if (ReflectionUtilities.matchesAnyPattern(element, patterns, matcher)) {
                lastIgnoreIndex = i;
                continue;
            }
            if (lastIgnoreIndex != -1) break;
        }
        if (lastIgnoreIndex == -1) {
            Msg.error(ReflectionUtilities.class, "Change call to ReflectionUtils.  Did not find the following patterns in the call stack: " + String.valueOf(patterns));
        }
        if (lastIgnoreIndex == trace.length - 1) {
            Msg.error(ReflectionUtilities.class, "Change call to ReflectionUtils. Call stack contains only ignored patterns: " + String.valueOf(patterns));
        }
        int startIndex = lastIgnoreIndex + 1;
        StackTraceElement[] updatedTrace = Arrays.copyOfRange(trace, startIndex, trace.length);
        t.setStackTrace(updatedTrace);
        return t;
    }

    public static Throwable createThrowableWithStackOlderThan(Class<?> ... classes) {
        List<String> patterns = Arrays.stream(classes).map(c -> c.getName()).collect(Collectors.toList());
        return ReflectionUtilities.createThrowableWithStackOlderThan(patterns, StackElementMatcher.EXACT_CLASS);
    }

    public static StackTraceElement[] movePastStackTracePattern(StackTraceElement[] trace, String pattern) {
        boolean foundIt = false;
        int desiredStartIndex = 0;
        for (int i = 0; i < trace.length; ++i) {
            StackTraceElement element = trace[i];
            boolean matches = ReflectionUtilities.matchesAnyPattern(element, pattern);
            if (foundIt && !matches) {
                desiredStartIndex = i;
                break;
            }
            foundIt |= matches;
        }
        if (!foundIt) {
            return trace;
        }
        StackTraceElement[] updatedTrace = Arrays.copyOfRange(trace, desiredStartIndex, trace.length);
        return updatedTrace;
    }

    public static StackTraceElement[] filterStackTrace(StackTraceElement[] trace, String ... patterns) {
        ArrayList<StackTraceElement> list = new ArrayList<StackTraceElement>();
        for (StackTraceElement element : trace) {
            if (ReflectionUtilities.matchesAnyPattern(element, patterns)) continue;
            list.add(element);
        }
        return list.toArray(new StackTraceElement[list.size()]);
    }

    public static Throwable createFilteredThrowable(String ... patterns) {
        Throwable t = ReflectionUtilities.createThrowableWithStackOlderThan(new ArrayList<String>());
        StackTraceElement[] trace = t.getStackTrace();
        StackTraceElement[] filtered = ReflectionUtilities.filterStackTrace(trace, patterns);
        t.setStackTrace(filtered);
        return t;
    }

    public static Throwable createJavaFilteredThrowable() {
        Throwable t = ReflectionUtilities.createThrowableWithStackOlderThan(List.of());
        return ReflectionUtilities.filterJavaThrowable(t);
    }

    public static String createJavaFilteredThrowableString() {
        Throwable t = ReflectionUtilities.createThrowableWithStackOlderThan(List.of());
        Throwable filtered = ReflectionUtilities.filterJavaThrowable(t);
        return ReflectionUtilities.stackTraceToString(filtered);
    }

    public static Throwable filterJavaThrowable(Throwable t) {
        StackTraceElement[] trace = t.getStackTrace();
        StackTraceElement[] filtered = ReflectionUtilities.filterStackTrace(trace, JAVA_AWT_PATTERN, JAVA_REFLECT_PATTERN, JDK_INTERNAL_REFLECT_PATTERN, SWING_JAVA_PATTERN, SWING_JAVAX_PATTERN, SECURITY_PATTERN, SUN_AWT_PATTERN, SUN_REFLECT_PATTERN, MOCKIT_PATTERN, JUNIT_PATTERN);
        t.setStackTrace(filtered);
        return t;
    }

    private static boolean matchesAnyPattern(StackTraceElement element, String ... patterns) {
        return ReflectionUtilities.matchesAnyPattern(element, List.of(patterns), StackElementMatcher.CONTAINS_ANY);
    }

    private static boolean matchesAnyPattern(StackTraceElement element, List<String> patterns, StackElementMatcher matcher) {
        for (String pattern : patterns) {
            if (!matcher.matches(element, pattern)) continue;
            return true;
        }
        return false;
    }

    public static String createStackTraceForAllThreads() {
        Map<Thread, StackTraceElement[]> allStackTraces = Thread.getAllStackTraces();
        Set<Map.Entry<Thread, StackTraceElement[]>> entrySet = allStackTraces.entrySet();
        StringBuilder builder = new StringBuilder();
        for (Map.Entry<Thread, StackTraceElement[]> entry : entrySet) {
            StackTraceElement[] value;
            builder.append("Thread: " + entry.getKey().getName()).append('\n');
            for (StackTraceElement stackTraceElement : value = entry.getValue()) {
                builder.append('\t').append("at ").append(stackTraceElement).append('\n');
            }
        }
        return builder.toString();
    }

    public static LinkedHashSet<Class<?>> getSharedHierarchy(List<?> list) {
        Object seed = list.get(0);
        LinkedHashSet master = new LinkedHashSet();
        Class<?> start = seed.getClass();
        boolean shareType = list.stream().allMatch(t -> t.getClass().equals(start));
        if (shareType) {
            master.add(start);
        }
        LinkedHashSet<Class<Class<?>>> parents = ReflectionUtilities.getAllParents(seed.getClass());
        Iterator<?> iterator = list.iterator();
        iterator.next();
        while (iterator.hasNext()) {
            Object o = iterator.next();
            LinkedHashSet<Class<?>> next = ReflectionUtilities.getAllParents(o.getClass());
            parents.retainAll(next);
        }
        master.addAll(parents);
        if (master.isEmpty()) {
            master.add(Object.class);
        }
        return master;
    }

    public static LinkedHashSet<Class<?>> getSharedParents(List<?> list) {
        Object seed = list.get(0);
        LinkedHashSet<Class<Class<?>>> master = ReflectionUtilities.getAllParents(seed.getClass());
        Iterator<?> iterator = list.iterator();
        iterator.next();
        while (iterator.hasNext()) {
            Object o = iterator.next();
            LinkedHashSet<Class<?>> next = ReflectionUtilities.getAllParents(o.getClass());
            master.retainAll(next);
        }
        if (master.isEmpty()) {
            master.add(Object.class);
        }
        return master;
    }

    public static String stackTraceToString(Throwable t) {
        return ReflectionUtilities.stackTraceToString(t.getMessage(), t);
    }

    public static String stackTraceToString(String message, Throwable t) {
        StringBuilder sb = new StringBuilder();
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        PrintStream ps = new PrintStream(baos);
        if (message != null) {
            ps.println(message);
        } else {
            String throwableMessage = t.getMessage();
            if (throwableMessage != null) {
                ps.println(throwableMessage);
            }
        }
        t.printStackTrace(ps);
        sb.append(baos.toString());
        ps.close();
        try {
            baos.close();
        }
        catch (IOException iOException) {
            // empty catch block
        }
        return sb.toString();
    }

    public static LinkedHashSet<Class<?>> getAllParents(Class<?> c) {
        LinkedHashSet l = new LinkedHashSet();
        if (Object.class.equals(c)) {
            l.add(Object.class);
            return l;
        }
        ReflectionUtilities.doGetAllParents(c, l);
        return l;
    }

    private static void doGetAllParents(Class<?> c, LinkedHashSet<Class<?>> accumulator) {
        Class<?>[] interfaces = c.getInterfaces();
        accumulator.addAll(Arrays.asList(interfaces));
        for (Class<?> clazz : interfaces) {
            ReflectionUtilities.doGetAllParents(clazz, accumulator);
        }
        Class<?> superclass = c.getSuperclass();
        if (superclass != null) {
            accumulator.add(superclass);
            ReflectionUtilities.doGetAllParents(superclass, accumulator);
        }
    }

    public static <T> List<Class<?>> getTypeArguments(Class<T> baseClass, Class<? extends T> childClass) {
        Objects.requireNonNull(baseClass);
        Objects.requireNonNull(childClass);
        HashMap<Type, Type> resolvedTypesDictionary = new HashMap<Type, Type>();
        Type baseClassAsType = ReflectionUtilities.walkClassHierarchyAndResolveTypes(baseClass, resolvedTypesDictionary, childClass);
        Type[] baseClassDeclaredTypeArguments = ReflectionUtilities.getDeclaredTypeArguments(baseClassAsType);
        return ReflectionUtilities.resolveBaseClassTypeArguments(resolvedTypesDictionary, baseClassDeclaredTypeArguments);
    }

    private static <T> Type walkClassHierarchyAndResolveTypes(Class<T> baseClass, Map<Type, Type> resolvedTypes, Type type) {
        if (type == null) {
            return null;
        }
        if (ReflectionUtilities.equals(type, baseClass)) {
            return type;
        }
        if (type instanceof Class) {
            Class clazz = (Class)type;
            Type[] interfaceTypes = clazz.getGenericInterfaces();
            HashSet<Type> toCheck = new HashSet<Type>();
            toCheck.addAll(Arrays.asList(interfaceTypes));
            Type parentType = clazz.getGenericSuperclass();
            toCheck.add(parentType);
            for (Type t : toCheck) {
                Type result = ReflectionUtilities.walkClassHierarchyAndResolveTypes(baseClass, resolvedTypes, t);
                if (!ReflectionUtilities.equals(result, baseClass)) continue;
                return result;
            }
            return parentType;
        }
        ParameterizedType parameterizedType = (ParameterizedType)type;
        Class rawType = (Class)parameterizedType.getRawType();
        Type[] actualTypeArguments = parameterizedType.getActualTypeArguments();
        TypeVariable<Class<T>>[] typeParameters = rawType.getTypeParameters();
        for (int i = 0; i < actualTypeArguments.length; ++i) {
            resolvedTypes.put(typeParameters[i], actualTypeArguments[i]);
        }
        if (rawType.equals(baseClass)) {
            return rawType;
        }
        Type[] interfaceTypes = rawType.getGenericInterfaces();
        HashSet<Type> toCheck = new HashSet<Type>();
        toCheck.addAll(Arrays.asList(interfaceTypes));
        Type parentType = rawType.getGenericSuperclass();
        toCheck.add(parentType);
        for (Type t : toCheck) {
            Type result = ReflectionUtilities.walkClassHierarchyAndResolveTypes(baseClass, resolvedTypes, t);
            if (!ReflectionUtilities.equals(result, baseClass)) continue;
            return result;
        }
        return parentType;
    }

    private static boolean equals(Type type, Class<?> c) {
        Class<?> typeClass = ReflectionUtilities.getClass(type);
        if (typeClass == null) {
            return false;
        }
        return typeClass.equals(c);
    }

    private static Class<?> getClass(Type type) {
        if (type instanceof Class) {
            return (Class)type;
        }
        if (type instanceof ParameterizedType) {
            return ReflectionUtilities.getClass(((ParameterizedType)type).getRawType());
        }
        if (type instanceof GenericArrayType) {
            GenericArrayType arrayType = (GenericArrayType)type;
            Type componentType = arrayType.getGenericComponentType();
            Class<?> componentClass = ReflectionUtilities.getClass(componentType);
            if (componentClass != null) {
                return Array.newInstance(componentClass, 0).getClass();
            }
            return null;
        }
        return null;
    }

    private static List<Class<?>> resolveBaseClassTypeArguments(Map<Type, Type> resolvedTypes, Type[] genericTypeArguments) {
        ArrayList typeArgumentsAsClasses = new ArrayList();
        for (Type baseType : genericTypeArguments) {
            while (resolvedTypes.containsKey(baseType)) {
                baseType = resolvedTypes.get(baseType);
            }
            typeArgumentsAsClasses.add(ReflectionUtilities.getClass(baseType));
        }
        return typeArgumentsAsClasses;
    }

    private static Type[] getDeclaredTypeArguments(Type type) {
        if (type instanceof Class) {
            return ((Class)type).getTypeParameters();
        }
        return ((ParameterizedType)type).getActualTypeArguments();
    }

    private static class StackElementMatcher {
        static final StackElementMatcher EXACT_CLASS = new StackElementMatcher(Match.EXACT, Content.CLASS_NAME);
        static final StackElementMatcher CONTAINS_CLASS_OR_METHOD = new StackElementMatcher(Match.CONTAINS, Content.CLASS_AND_METHOD_NAME);
        static final StackElementMatcher CONTAINS_ANY = new StackElementMatcher(Match.CONTAINS, Content.ALL);
        private Match matchType;
        private Content contentType;

        StackElementMatcher(Match matchType, Content contentType) {
            this.matchType = matchType;
            this.contentType = contentType;
        }

        boolean matches(StackTraceElement element, String pattern) {
            String s = this.contentType.convert(element);
            return this.matchType.matches(s, pattern);
        }

        static enum Match {
            EXACT,
            CONTAINS;


            boolean matches(String input, String pattern) {
                switch (this.ordinal()) {
                    case 0: {
                        return input.equals(pattern);
                    }
                    case 1: {
                        return input.contains(pattern);
                    }
                }
                throw new AssertException("Missing case type");
            }
        }

        static enum Content {
            CLASS_NAME,
            CLASS_AND_METHOD_NAME,
            ALL;


            String convert(StackTraceElement e) {
                switch (this.ordinal()) {
                    case 0: {
                        return e.getClassName();
                    }
                    case 1: {
                        return e.getClassName() + " " + e.getMethodName();
                    }
                    case 2: {
                        return e.toString();
                    }
                }
                throw new AssertException("Missing case type");
            }
        }
    }
}

